// Example4 is    This is a Traveling Salesman Problem with 
// 7 cities located in a 10 X 10 square:
//
// The cities have the following (x,y) coordinates within the 10 X 10 square.
//
//                         X       Y
//		1	Andover			5.18		5.72
//		2	Bellows Falls	3.07		8.14
//		3	Chester			0.72		3.79
//		4	Dorset			1.66		5.25
//		5	Elizabethville	4.65		8.57
//		6	Fonda				6.51		2.01
//		7	Galeton			1.83		0.08
//
// Optimization problem is to visit all 15 cities, traveling the 
// shortest distance.
//
// The OptQuest.dll distributed with this example is a demo
// version with the following restrictions:
//		- Maximum variables = 7	
//		- Maximum constraints = 4
//		- Maximum requirements =  4
//		- Maximum iterations = 500
//

#include "ocl.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#define TOUR 7

double evaluate(double *);

int main(int argc,char *argv[])
{
	double tour[TOUR+1];
	double tourlength;
	long   nprob, nsol, status;
	int    i, TotalIter;

	if (argc != 2) 
	{
		TotalIter = 500;
	}
	else
		TotalIter = atoi(argv[1]);

	if (TotalIter > 500)
	{
		printf("Warning:  The Demo version is limited to 500 iterations\n");
		TotalIter = 500;
	}


	/* Allocating memory */
	nprob = OCLSetup(0,TOUR,0,0,"MIN", 999999999);
	if (nprob < 0) 
	{
		printf("OCLSetup error code %d\n", nprob);
		exit(1);
	}

	/* Suggest a solution and initialize the population */
	for (i = 1; i <= TOUR; i++) tour[i] = (double)i;
	status = OCLSugPopSolution(nprob, tour);
	if (status < 0) 
	{
		printf("OCLSugPopSolution error code %d\n", status);
		exit(1);
	} 
	status = OCLInitPop(nprob);
	if (status < 0) 
	{
		printf("OCLInitPop error code %d\n", status);
		exit(1);
	}

	/* Generate and evaluate TotalIter solutions */
	for (i = 1; i <= TotalIter; i++)
	{
		nsol = OCLGetSolution(nprob,tour);
		if (nsol < 0) 
		{
			printf("OCLGetSolution error code %d\n", nsol);
			exit(1);
		}
		tourlength = evaluate(tour);
		status = OCLPutSolution(nprob,nsol,&tourlength,(double *)OCLGetNull(nprob));
		if (status < 0) 
		{
			printf("OCLPutSolution error code %d\n", status);
			exit(1);
		}
		if (!(i%1000)) 
		{
			status = OCLGetBest(nprob,tour,&tourlength);
			if (status < 0) {
				printf("OCLGetBest error code %d\n", status);
				exit(1);
			}
			printf("Best solution after %7d iterations is %9.6f\n",i,tourlength);
		}
	}

	/* Display the best solution found */
	status = OCLGetBest(nprob,tour,&tourlength);
	if (status < 0) 
	{
		printf("OCLGetBest error code %d\n", status);
		exit(1);
	}

	printf("Tour length = %9.6f\n", tourlength);

	printf("\nTour: ");
	for (i = 1; i <= TOUR; i++) printf("%4.0lf", tour[i]);
	printf("\nPress any key to end ...\n");
	getchar();

	/* Free memory */
	status = OCLGoodBye(nprob);
	if (status < 0) 
	{
		printf("OCLGoodBye error code %d\n", status);
		exit(1);
	}
	return 0;
}

/* Evaluation function */

double evaluate(double *tour)
{
	int i;
	double x[TOUR+1], y[TOUR+1];
	double tourlength;

	x[1]  = 5.18;	y[1]  = 5.72;
	x[2]  = 3.07;	y[2]  = 8.14;
	x[3]  = 0.72;	y[3]  = 3.79;
	x[4]  = 1.66;	y[4]  = 5.25;
	x[5]  = 4.65;	y[5]  = 8.57;
	x[6]  = 6.51;	y[6]  = 2.01;
	x[7]  = 1.83;	y[7]  = 0.08;

	tourlength = pow(pow(x[(int)tour[TOUR]]-x[(int)tour[1]],2)+
			         pow(y[(int)tour[TOUR]]-y[(int)tour[1]],2),0.5);
	for (i = 1; i <= TOUR-1; ++i)
		tourlength += pow(pow(x[(int)tour[i]]-x[(int)tour[i+1]],2)+
		                  pow(y[(int)tour[i]]-y[(int)tour[i+1]],2),0.5);

	return tourlength;
}
